clear all

* Set Folder to obtain input database

cd ..\Inputs

* Import WEO's

use StataWEO, clear

* Set Folder to save output database

cd ..\Outputs

* Obtain variable names

unab mylist: y*v20162
foreach v of local mylist {
	local stubs `"`stubs' `=substr("`v'",1,6)'"'
} 
reshape long `stubs', i(Country WEOSubjectCode) j(year)

* Relation between varnum and variables
 			 
* 1  = LE           Employment
* 2  = LUR          Unemployment Rate 
* 3  = NGAP_NPGDP   Output Gap in % of Potential GDP
* 4  = NGDP         Nominal GDP in Local Currency 
* 5  = NGDPD        Nominal GDP in Dollars 
* 6  = NGDP_D       Nominal GDP deflator
* 7  = NGDP_R       Real GDP in local currency units (different base years)
* 8  = NGSD_NGDP    Gross National Saving as % of GDP
* 9  = NID_NGDP     Investment as a % of GDP
* 10 = PCPI         Inflation According to CPI 
			 
drop WEOSubjectCode
reshape wide `stubs', i(Country year) j(varnum)
			 
* What we have now is a dataset with country, year-semester that indicates
* the moment in time when the WEO was published and a number of variables 
* y`x'v`z', where `x' represents the point in time that the data is describing 
* and `z' represents a varnum as described above (which type of variable)

* Generate date indicators

tostring year, replace
gen semester=substr(year,5,1)
destring semester, replace
replace year=substr(year,1,4)
destring year, replace

* Rename numeric varnums to more understandable variable names, according
* to relation betwen varnum and variable type described above.

forvalues i=1980(1)2021 {
	rename y`i'v1  emp`i'
    rename y`i'v2  unr`i'
	rename y`i'v3  gap`i'
    rename y`i'v4  gdplcur`i'
    rename y`i'v5  output`i'
	rename y`i'v6  gdpdefl`i'
	rename y`i'v7  realgdp`i'
    rename y`i'v8  gns`i'
	rename y`i'v9  inv`i'
    rename y`i'v10 cpi`i'
}

* Generate potential output by combining output gap and GDP

forvalues i=1980(1)2021 {
	gen Plcu`i'=(realgdp`i')/((gap`i'/100)+1)
	gen Ylcu`i'=realgdp`i'
}

* Rearrange and save

order Country ISO year semester
save Potential, replace

* Set Panel

by Country: gen tt=_n
xtset Country tt

* Generate Shocks to potential output
* First generate log potential output in lcu

quiet forvalues i=1980(1)2021 {
	gen LPlcu`i'=ln(Plcu`i')
	gen LYlcu`i'=ln(Ylcu`i')
}

* Then generate the change in log potential output from one year to the 
* next (over actual time, not over publications of WEO)

forvalues i=1981(1)2021 {
	gen GPlcu`i'=LPlcu`i'-LPlcu`=`i'-1'
	gen GYlcu`i'=LYlcu`i'-LYlcu`=`i'-1'
}

* Then generate the revisions to these growth rates across editions of the WEO's

forvalues i=1981(1)2021 {
	gen DGPlcu`i'=d.GPlcu`i'
}

* Assign the appropriate revision to each time period

gen SC=.
gen S4=.
forvalues i=2004(1)2016 {
	replace SC=DGPlcu`i' if year==`i' & semester==1
	capture drop S4int1 miss1
	egen    miss1 =rowmiss(DGPlcu`i'-DGPlcu`=`i'+3')
	egen    S4int1=rowtotal(DGPlcu`i'-DGPlcu`=`i'+3')
	replace S4int1=. if miss1>0
	replace S4=S4int1 if year==`i' & semester==1
	replace SC=DGPlcu`=`i'+1' if year==`i' & semester==2
	capture drop S4int2 miss2
	egen    miss2 =rowmiss(DGPlcu`=`i'+1'-DGPlcu`=`i'+4')
	egen    S4int2=rowtotal(DGPlcu`=`i'+1'-DGPlcu`=`i'+4')
	replace S4int2=. if miss2>0
	replace S4=S4int2 if year==`i' & semester==2
}

* Assign the appropriate growth rates of output to each time period

foreach yp in Y P {
	forvalues r=0(1)4 {
		forvalues j=0(1)1 {
			gen G`yp'IMFtp`r'gtm`j'=.
			forvalues i=2003(1)2016 {
				replace G`yp'IMFtp`r'gtm`j'=l`j'.G`yp'lcu`=`i'+`r'' if year==`i' & semester==1
				replace G`yp'IMFtp`r'gtm`j'=l`j'.G`yp'lcu`=`i'+`r'+1' if year==`i' & semester==2
			}
		}
		forvalues j=1(1)3 {
			gen G`yp'IMFtp`r'gtp`j'=.
			forvalues i=2003(1)2016 {
				replace G`yp'IMFtp`r'gtp`j'=f`j'.G`yp'lcu`=`i'+`r'' if year==`i' & semester==1
				replace G`yp'IMFtp`r'gtp`j'=f`j'.G`yp'lcu`=`i'+`r'+1' if year==`i' & semester==2
			}
		}
	}
	forvalues i=1981(1)2021 {
		gen G`yp'IMFlast`i'=.
		replace G`yp'IMFlast`i'=G`yp'lcu`i' if year==2016 & semester==2
	}
	forvalues r=0(1)4 {
		gen G`yp'IMFtp`r'gT=.
		forvalues i=1981(1)2016 {
			capture drop G`yp'IMFtp`r'gTtemp
			by Country: egen G`yp'IMFtp`r'gTtemp=mean(G`yp'IMFlast`=`i'+`r'+1')
			replace G`yp'IMFtp`r'gT=G`yp'IMFtp`r'gTtemp if year==`i' & semester==2
		}
	}
	drop G`yp'IMFtp?gTtemp G?IMFlast*
}

keep ISO year semester SC S4 G?IMFtp?g*
save IMFREV, replace

* Import Database again

use Potential, clear

* Rename time variables to denote that they correspond to WEO editions

rename year     yearWEO
rename semester semesterWEO

* Keep only relevant variables

keep Country ISO yearWEO semesterWEO Plcu* Ylcu*

* Reshape to have calendar year as an additional variable

reshape long Plcu Ylcu, i(Country ISO yearWEO semesterWEO) j(year)

* Generate an identification number for panel and set panel

egen id=group(Country yearWEO semesterWEO)
xtset id year

* Generate growth rates of potential and actual

foreach v in P Y {
	gen G`v'lcu=(`v'lcu-L.`v'lcu)/L.`v'lcu*100
}

* Generate HP filtered actual output

local i=1
foreach lambda of numlist 3 6.25 10 15 25 50 100 200 400 800 {
	di `lambda'
	di `i'
	tsfilter hp hpcGY`i' = GYlcu, smooth(`lambda') trend(hpGY`i')
	drop hpc*`i'
	label var hpGY`i' "GYlcu trend component from HP with lambda=`lambda'"
	local ++i
}

* Generate MA filtered actual output   
	   
forvalues terms=1(1)10 {
	di `terms'
	tssmooth ma maGY`terms'=GYlcu, window(`=`terms'-1' 1 0)
	label var maGY`terms' "GYlcu filtered with MA(`terms')"
}

forvalues terms=1(1)10 {
	quiet replace maGY`terms'=. if GYlcu==.
}

* Keep only the appropriate observations

gen     keepmau=0
replace keepmau=1 if year==yearWEO+1 & semesterWEO==2
replace keepmau=1 if year==yearWEO   & semesterWEO==1 
keep if keepmau==1

* Drop and rename

drop Country year id keepmau Plcu Ylcu
rename yearWEO year
rename semesterWEO semester
rename *G?* *G?IMF*

* Save database and erase temporary dataset

save IMFHPMA, replace
erase Potential.dta
